//	TorusGames2DApples.c
//
//	Apples2D coordinate conventions
//
//	Integer coordinates (h,v) label the cells, for h,v ∈ {0, 1, … , n-1}.
//	The first coordinate runs left-to-right, while the second runs bottom-to-top.
//	Cell centers align with the natural n×n grid, so for example
//	cells straddle the edges of the fundamental domain
//	and a single cell sits centered at the fundamental domain's corners.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesLocalization.h"
#include "GeometryGamesSound.h"
#include <math.h>
#ifdef GAME_CONTENT_FOR_SCREENSHOT
#include <stdio.h>	//	for printf()
#endif


#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_APPLES	11

//	The body of the apple isn't quite centered vertically in the cell,
//	so the smiley face marker shouldn't be either.
#define APPLE_MARK_OFFSET	0.06

//	A quick tap eats an apple, but
//	a long touch marks an apple as wormy.
//	How long must a long touch be?
#define APPLES_MIN_LONG_PRESS_DURATION	0.1875	//	in seconds

//	Don't clear itsShowBriefInstructionsFlag until the user
//	has successfully marked a few worms.
//	Otherwise there's a small chance that the user could
//	get a very easy board that "blows out" on the first few taps,
//	without the user having marked any worms at all.
#define NUM_MARKED_WORMS_TO_HIDE_INSTRUCTIONS	3


//	A queue of CellCoordinates will keep track of which
//	apples have been recently uncovered.
typedef struct
{
	unsigned int	itsH,
					itsV;
} CellCoordinates;


//	Public functions with private names
static void	ApplesReset(ModelData *md);
static void	ApplesHandMoved(ModelData *md);
static bool	ApplesDragBegin(ModelData *md, bool aRightClick);
static void	ApplesDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void	ApplesDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
static void	ApplesRefreshMessage(ModelData *md);

//	Private functions
static Byte	CountNeighbors(ModelData *md, unsigned int h, unsigned int v);
static void	FindCurrentApple(ModelData *md, unsigned int *h, unsigned int *v, bool *aFlip);
static void	DeclareLoss(ModelData *md);
static bool	PlayerHasWon(ModelData *md);
static void	DeclareWin(ModelData *md);
static void	ExposeHitCells(ModelData *md, unsigned int h, unsigned int v);


void Apples2DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &ApplesReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= &ApplesHandMoved;
	md->itsGame2DDragBegin				= &ApplesDragBegin;
	md->itsGame2DDragObject				= &ApplesDragObject;
	md->itsGame2DDragEnd				= &ApplesDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= NULL;
	md->itsGameRefreshMessage			= &ApplesRefreshMessage;
	
	//	Set up the board.
	ApplesReset(md);
	
	//	Display the brief instructions, for the sake of first-time users.
	md->itsGameOf.Apples2D.itsShowBriefInstructionsFlag				= true;
	md->itsGameOf.Apples2D.itsShowBriefInstructionsMarkedWormCount	= 0;
	ApplesRefreshMessage(md);
}


const Char16 *ChooseApplesNumeralFont(void)
{
	//	Choose a font that looks good and is available on the host platform.

#ifdef __APPLE__	//	iOS and macOS
	return u"Helvetica";
#else				//	some other platform
#error Would need to select a font for the new platform.
#endif
}


static void ApplesReset(ModelData *md)
{
	unsigned int	theNumCells,
					theNumWorms,
					theNumRemainingCells,
					theNumRemainingWorms,
					h,
					v;

#ifdef GAME_CONTENT_FOR_SCREENSHOT
#if 0	//	experiment with different seeds, to see which looks good?
#define EXPOSE_GOOD_APPLES
	{
		static unsigned int	theSeed	= 0;

		RandomInitWithSeed(theSeed);
		printf("using Apples seed %d\n", theSeed);
		theSeed++;
	}
#else	//	use our favorite seed
	{
		//	Caution:  The random number generator's implementation is different
		//	on different platforms, so while a given seed generates the same jigsaw puzzle
		//	on iOS and macOS, we'd need to use a different seed on other platforms.
		RandomInitWithSeed(9);
	}
#endif	//	experiment or use favorite seed
#endif	//	GAME_CONTENT_FOR_SCREENSHOT

	//	Set the board size.
	switch (md->itsDifficultyLevel)
	{
		//	Board sizes had been {4, 8, 16, 32}.
		//	Changed to odd values to better show Klein bottle axes.
		case 0:  md->itsGameOf.Apples2D.itsBoardSize =  5;  break;
		case 1:  md->itsGameOf.Apples2D.itsBoardSize =  7;  break;
		case 2:  md->itsGameOf.Apples2D.itsBoardSize = 15;  break;
		case 3:  md->itsGameOf.Apples2D.itsBoardSize = 31;  break;
		default: md->itsGameOf.Apples2D.itsBoardSize =  2;  break;	//	should never occur
	}
#ifdef MAKE_GAME_CHOICE_ICONS
	md->itsGameOf.Apples2D.itsBoardSize = 6;
#endif

	//	Set a fixed number of worms.
	//	When the number of worms is roughly 1/7 the number of cells,
	//	the game is almost always winnable, but usually requires some thought.
	//	Challenging enough to be fun, but not super hard.
	theNumCells	= md->itsGameOf.Apples2D.itsBoardSize * md->itsGameOf.Apples2D.itsBoardSize;
	theNumWorms	= theNumCells / 7;	//	integer division discards remainder

	//	Clear the board.
	for (h = 0; h < MAX_APPLE_BOARD_SIZE; h++)
		for (v = 0; v < MAX_APPLE_BOARD_SIZE; v++)
			md->itsGameOf.Apples2D.itsBoard[h][v] = 0x00;

	//	Just to be safe...
	if (md->itsGameOf.Apples2D.itsBoardSize > MAX_APPLE_BOARD_SIZE)
		return;

#ifdef MAKE_GAME_CHOICE_ICONS
	//	Place a few worms in the lower left corner.
	//
	//		To make the icon, expose the cells at (1,0), (1,1) and (2,1),
	//		mark the apples at (0,1) and (1,2), then export the whole board
	//		at 2n×2n pixels, and copy out the n×n pixel block
	//		in the board's lower-left quadrant, for n ∈ {32, 64, 96}.
	//
	md->itsGameOf.Apples2D.itsBoard[0][1] |= APPLE_HAS_WORM_MASK;
	md->itsGameOf.Apples2D.itsBoard[1][5] |= APPLE_HAS_WORM_MASK;
	md->itsGameOf.Apples2D.itsBoard[1][2] |= APPLE_HAS_WORM_MASK;
	md->itsGameOf.Apples2D.itsBoard[3][3] |= APPLE_HAS_WORM_MASK;
	md->itsGameOf.Apples2D.itsBoard[4][0] |= APPLE_HAS_WORM_MASK;
#else
	//	Assign worms to apples in such a way that
	//
	//		the total number of worms comes out right
	//	and
	//		all apples have the same chance of getting a worm.
	//
	theNumRemainingCells = theNumCells;
	theNumRemainingWorms = theNumWorms;
	for (h = 0; h < md->itsGameOf.Apples2D.itsBoardSize; h++)
	{
		for (v = 0; v < md->itsGameOf.Apples2D.itsBoardSize; v++)
		{
			if ((RandomUnsignedInteger() % theNumRemainingCells) < theNumRemainingWorms)
			{
				md->itsGameOf.Apples2D.itsBoard[h][v] |= APPLE_HAS_WORM_MASK;
				theNumRemainingWorms--;
			}
			theNumRemainingCells--;
		}
	}
#endif

#ifdef GAME_CONTENT_FOR_SCREENSHOT
#ifdef EXPOSE_GOOD_APPLES
	for (h = 0; h < md->itsGameOf.Apples2D.itsBoardSize; h++)
	{
		for (v = 0; v < md->itsGameOf.Apples2D.itsBoardSize; v++)
		{
			if ( ! (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_HAS_WORM_MASK) )
			{
				md->itsGameOf.Apples2D.itsBoard[h][v] |= APPLE_EXPOSED_MASK;
			}
		}
	}
#endif
#endif

	//	Count each square's neighbors.
	//	(In a 2×2 or smaller board, some neighbors will get counted more than once.)
	for (h = 0; h < md->itsGameOf.Apples2D.itsBoardSize; h++)
		for (v = 0; v < md->itsGameOf.Apples2D.itsBoardSize; v++)
			md->itsGameOf.Apples2D.itsBoard[h][v] |= CountNeighbors(md, h, v);

	//	In a Klein bottle, assign flips randomly.
	if (md->itsTopology == Topology2DKlein)
		for (h = 0; h < md->itsGameOf.Apples2D.itsBoardSize; h++)
			for (v = 0; v < md->itsGameOf.Apples2D.itsBoardSize; v++)
				if (RandomBoolean())
					md->itsGameOf.Apples2D.itsBoard[h][v] |=  APPLE_FLIPPED_MASK;

#ifdef GAME_CONTENT_FOR_SCREENSHOT
	//	These simulated taps assume seed #9 above.
	ExposeHitCells(md, 2, 2);
	ExposeHitCells(md, 2, 4);
	ExposeHitCells(md, 4, 4);
	ExposeHitCells(md, 5, 4);
	ExposeHitCells(md, 5, 1);
	md->itsGameOf.Apples2D.itsBoard[0][1] |= APPLE_MARKED_MASK;
	md->itsGameOf.Apples2D.itsBoard[0][3] |= APPLE_MARKED_MASK;
	md->itsGameOf.Apples2D.itsBoard[3][4] |= APPLE_MARKED_MASK;
	md->itsGameOf.Apples2D.itsBoard[5][3] |= APPLE_MARKED_MASK;
#endif

	//	Abort any pending simulation.
	SimulationEnd(md);

	//	Ready to go.
	md->itsGameIsOver = false;
}


static Byte CountNeighbors(
	ModelData		*md,
	unsigned int	h,
	unsigned int	v)
{
	Byte		theCount;
	signed int	dh,
				dv,
				hh,
				vv,
				hhh,
				vvv;

	theCount = 0;

	for (dv = -1; dv <= +1; dv++)
	{
		hh = h;
		vv = v + dv;
		if (vv < 0)
		{
			vv = md->itsGameOf.Apples2D.itsBoardSize - 1;
			if (md->itsTopology == Topology2DKlein)
				hh = (hh > 0) ? md->itsGameOf.Apples2D.itsBoardSize - hh : 0;
		}
		if (vv > (signed int)(md->itsGameOf.Apples2D.itsBoardSize - 1))
		{
			vv = 0;
			if (md->itsTopology == Topology2DKlein)
				hh = (hh > 0) ? md->itsGameOf.Apples2D.itsBoardSize - hh : 0;
		}

		for (dh = -1; dh <= +1; dh++)
		{
			if (dh == 0 && dv == 0)
				continue;

			hhh = hh + dh;
			vvv = vv;
			if (hhh < 0)
				hhh = md->itsGameOf.Apples2D.itsBoardSize - 1;
			if (hhh > (signed int)(md->itsGameOf.Apples2D.itsBoardSize - 1))
				hhh = 0;

			if (md->itsGameOf.Apples2D.itsBoard[hhh][vvv] & APPLE_HAS_WORM_MASK)
				theCount++;
		}
	}

	return theCount;
}


static void ApplesHandMoved(ModelData *md)
{
	unsigned int	h,
					v;
	bool			theFlip;

	//	Which apple is the hand cursor over?
	FindCurrentApple(md, &h, &v, &theFlip);

	//	The orientation of the symbol relative to the fundamental domain
	//	should match the current orientation of the hand relative 
	//	to the fundamental domain.
	if (theFlip)
		md->itsGameOf.Apples2D.itsBoard[h][v] |=  APPLE_FLIPPED_MASK;
	else
		md->itsGameOf.Apples2D.itsBoard[h][v] &= ~APPLE_FLIPPED_MASK;
}


static bool ApplesDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	unsigned int	h,
					v;

	//	If the game is already over, ignore all hits.
	if (md->itsGameIsOver)
		return false;

	//	Which apple got hit?
	FindCurrentApple(md, &h, &v, NULL);

	//	If the apple has alreay been eaten, treat this hit as a scroll.
	if (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_EXPOSED_MASK)
		return false;

	//	During a content drag, remember which cell was hit originally,
	//	and accept the hit iff the drag ends over that same cell.
	//	This gives the user a chance to change his/her mind,
	//	and also helps protect against accidentally eating an apple
	//	when trying to scroll.
	md->itsGameOf.Apples2D.itsHitCellH = h;
	md->itsGameOf.Apples2D.itsHitCellV = v;
	
	//	Remember whether this was a right click.
	md->itsGameOf.Apples2D.itsRightClickFlag = aRightClick;

	return true;
}


static void ApplesDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	md->its2DHandPlacement.itsH += (md->its2DHandPlacement.itsFlip ? -aHandLocalDeltaH : aHandLocalDeltaH);
	md->its2DHandPlacement.itsV += aHandLocalDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);
}


static void ApplesDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	unsigned int	h,
					v;

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized),
	//	return without eating or marking the apple.
	if (aTouchSequenceWasCancelled)
		return;

	//	In which cell did the drag end?
	FindCurrentApple(md, &h, &v, NULL);

	//	Accept the drag iff it ended in the same cell where it began.
	if (h == md->itsGameOf.Apples2D.itsHitCellH
	 && v == md->itsGameOf.Apples2D.itsHitCellV)
	{
		//	Is this a right-click or is the control key down?
		//
		//	Adam suggested an illogical yet user-friendly special case:
		//	if an apple is marked, clicking it should merely unmark it, not expose it.
		//
		if (md->itsGameOf.Apples2D.itsRightClickFlag						//	typical case (desktop)
		 || (aDragDuration > APPLES_MIN_LONG_PRESS_DURATION)				//	typical case (mobile)
		 || (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_MARKED_MASK))	//	special case
		{
			//	Toggle the mark.
			md->itsGameOf.Apples2D.itsBoard[h][v] ^= APPLE_MARKED_MASK;
			
			//	Count how many worms the user has marked,
			//	so we'll know when to hide the brief instruction message.
			md->itsGameOf.Apples2D.itsShowBriefInstructionsMarkedWormCount++;
		}
		else
		{
			//	Expose the cell.
			//	Did the player hit a worm?
			if (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_HAS_WORM_MASK)
				DeclareLoss(md);
			else
			{
				ExposeHitCells(md, h, v);

				if (PlayerHasWon(md))
					DeclareWin(md);
			}
		}
	}
}


static void FindCurrentApple(
	ModelData		*md,	//	input
	unsigned int	*h,		//	output
	unsigned int	*v,		//	output
	bool			*aFlip)	//	output, may be NULL
{
	double	theHandH,
			theHandV;

	//	Convert hand coordinates from [-0.5, +0.5] to [0.0, itsGameOf.Apples2D.itsBoardSize].
	theHandH = md->itsGameOf.Apples2D.itsBoardSize*(md->its2DHandPlacement.itsH + 0.5);
	theHandV = md->itsGameOf.Apples2D.itsBoardSize*(md->its2DHandPlacement.itsV + 0.5);

	//	Find the integer coordinates (h,v) of the hit cell.
	//	Results will lie in the range [0, n],
	//	but n will wrap to 0 immediately below.
	*h = (unsigned int) floor(theHandH + 0.5);
	*v = (unsigned int) floor(theHandV + 0.5);
	
	//	If the chirality of the hit cell is to agree with
	//	the chirality of the hand, should the hit cell be flipped?
	if (aFlip != NULL)
		*aFlip = md->its2DHandPlacement.itsFlip;

	//	Wrap values on the boundary and
	//	protect against unexpected errors.
	if (*v >= md->itsGameOf.Apples2D.itsBoardSize)
	{
		*v = 0;
		if (md->itsTopology == Topology2DKlein)
		{
			*h = md->itsGameOf.Apples2D.itsBoardSize - *h;
			if (aFlip != NULL)
				*aFlip = ! *aFlip;
		}
	}
	if (*h >= md->itsGameOf.Apples2D.itsBoardSize)
		*h = 0;
}


static void DeclareLoss(ModelData *md)
{
	unsigned int	h,
					v;

	md->itsGameIsOver = true;

	for (h = 0; h < md->itsGameOf.Apples2D.itsBoardSize; h++)
		for (v = 0; v < md->itsGameOf.Apples2D.itsBoardSize; v++)
			md->itsGameOf.Apples2D.itsBoard[h][v] |= APPLE_EXPOSED_MASK;

	EnqueueSoundRequest(u"ApplesLoss.mid");
}


static bool PlayerHasWon(ModelData *md)
{
	unsigned int	h,
					v;

	for (h = 0; h < md->itsGameOf.Apples2D.itsBoardSize; h++)
	{
		for (v = 0; v < md->itsGameOf.Apples2D.itsBoardSize; v++)
		{
			if (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_HAS_WORM_MASK)
			{
				if ((md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_EXPOSED_MASK) != 0)
					return false;
			}
			else
			{
				if ((md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_EXPOSED_MASK) == 0)
					return false;
			}
		}
	}

	return true;
}


static void DeclareWin(ModelData *md)
{
	unsigned int	h,
					v;

	md->itsGameIsOver = true;

	for (h = 0; h < md->itsGameOf.Apples2D.itsBoardSize; h++)
		for (v = 0; v < md->itsGameOf.Apples2D.itsBoardSize; v++)
			if (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_HAS_WORM_MASK)
				md->itsGameOf.Apples2D.itsBoard[h][v] |= APPLE_MARKED_MASK;

	EnqueueSoundRequest(u"ApplesWin.mid");

#ifdef GAME_CONTENT_FOR_SCREENSHOT
	//	Always show the brief instructions message while making screenshots.
#else
	//	After the user has marked a few worms and successfully won a game,
	//	s/he may no longer want to see the instructions.
	if (md->itsGameOf.Apples2D.itsShowBriefInstructionsMarkedWormCount > NUM_MARKED_WORMS_TO_HIDE_INSTRUCTIONS)
	{
		md->itsGameOf.Apples2D.itsShowBriefInstructionsFlag = false;
		ApplesRefreshMessage(md);
	}
#endif
}


static void ExposeHitCells(
	ModelData		*md,
	unsigned int	h,
	unsigned int	v)
{
	//	Keep a queue of all cells that get uncovered.
	//	If a newly uncovered cell has zero wormy neighbors,
	//	uncover any still-covered neighbors and add them
	//	to the queue as well.

	CellCoordinates	*theQueue		= NULL,
					*theQueueFirst	= NULL,
					*theQueueLast	= NULL;
	signed int		dh,
					dv,
					hh,
					vv,
					hhh,
					vvv;

	//	Uncover the original hit cell.
	md->itsGameOf.Apples2D.itsBoard[h][v] |= APPLE_EXPOSED_MASK;

#ifdef MAKE_GAME_CHOICE_ICONS
	//	When making the game choice icon,
	//	don't recursively expose cells.
	return;
#endif

	//	Allocate space for the queue.
	//	If we can't get it, fail silently.
	theQueue = GET_MEMORY(md->itsGameOf.Apples2D.itsBoardSize * md->itsGameOf.Apples2D.itsBoardSize * sizeof(CellCoordinates));
	if (theQueue == NULL)
		return;

	//	Put the original hit cell on the queue.
	theQueueFirst		= theQueue;
	theQueueLast		= theQueue;
	theQueueFirst->itsH	= h;
	theQueueFirst->itsV	= v;

	//	Process each newly uncovered cell.  If the cell has no wormy neighbors,
	//	uncover all previously covered neighbors and add them to the queue.
	while (theQueueFirst <= theQueueLast)
	{
		//	Pull the first apple off the queue and check its neighbors,
		//	uncovering them and adding them to the queue as necessary.
		if ((md->itsGameOf.Apples2D.itsBoard[theQueueFirst->itsH][theQueueFirst->itsV] & APPLE_NBR_COUNT_MASK) == 0)
		{
			for (dv = -1; dv <= +1; dv++)
			{
				hh = theQueueFirst->itsH;
				vv = theQueueFirst->itsV + dv;
				if (vv < 0)
				{
					vv = md->itsGameOf.Apples2D.itsBoardSize - 1;
					if (md->itsTopology == Topology2DKlein)
						hh = (hh > 0) ? md->itsGameOf.Apples2D.itsBoardSize - hh : 0;
				}
				if (vv > (signed int)(md->itsGameOf.Apples2D.itsBoardSize - 1))
				{
					vv = 0;
					if (md->itsTopology == Topology2DKlein)
						hh = (hh > 0) ? md->itsGameOf.Apples2D.itsBoardSize - hh : 0;
				}

				for (dh = -1; dh <= +1; dh++)
				{
					if (dh == 0 && dv == 0)
						continue;

					hhh = hh + dh;
					vvv = vv;
					if (hhh < 0)
						hhh = md->itsGameOf.Apples2D.itsBoardSize - 1;
					if (hhh > (signed int)(md->itsGameOf.Apples2D.itsBoardSize - 1))
						hhh = 0;

					if ((md->itsGameOf.Apples2D.itsBoard[hhh][vvv] & APPLE_EXPOSED_MASK) == 0)
					{
						md->itsGameOf.Apples2D.itsBoard[hhh][vvv] |= APPLE_EXPOSED_MASK;
						theQueueLast++;
						theQueueLast->itsH = hhh;
						theQueueLast->itsV = vvv;
					}
				}
			}
		}

		//	Move on.
		theQueueFirst++;
	}

	//	Free the queue.
	FREE_MEMORY_SAFELY(theQueue);
}


static void ApplesRefreshMessage(ModelData *md)
{
	const Char16	*theMessage;
	
	if (md->itsGameOf.Apples2D.itsShowBriefInstructionsFlag)
	{
		//	Let the user know how to mark apples as wormy.
		//	Here we display only the very brief version of the instructions;
		//	the Help page explains the game in full detail.
		theMessage =
#if TARGET_OS_IOS
									GetLocalizedText(u"Apples instructions - touch");
#else
									GetLocalizedText(u"Apples instructions - mouse");
#endif
	}
	else
	{
		theMessage = u"";
	}

	SetTorusGamesStatusMessage(	theMessage,
								(ColorP3Linear) {0.5, 0.0, 0.0, 1.0},	//	premultiplied alpha
								Game2DApples);
}


unsigned int GetNum2DApplesBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_APPLES;
}

void Get2DApplesKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 0.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 1.00;
	someKleinAxisColors[1][1] = 1.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DApplesSprites(
	unsigned int	aBoardSize)
{
	return aBoardSize * aBoardSize	//	full-size cells for apples, worms and numbers
		 + aBoardSize * aBoardSize;	//	half-size cells (with offset) for happy and yucky faces
}

void Get2DApplesSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int	n;				//	board size is n cells by n cells
	double			theCellSize;	//	1/n
	unsigned int	h,
					v,
					i;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DApples,
		"Game2DApples must be active");

	n			= md->itsGameOf.Apples2D.itsBoardSize;
	theCellSize	= 1.0 / (double) n;

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DApplesSprites(n),
		"Internal error:  wrong buffer size");

	//	full-size cells
	//		note (h,v) indexing -- not (row,col)
	for (h = 0; h < n; h++)
	{
		for (v = 0; v < n; v++)
		{
			i = n*h + v;
			aPlacementBuffer[i].itsH		= -0.5 + h * theCellSize;
			aPlacementBuffer[i].itsV		= -0.5 + v * theCellSize;
			aPlacementBuffer[i].itsFlip		= (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_FLIPPED_MASK);
			aPlacementBuffer[i].itsAngle	= 0.0;
			aPlacementBuffer[i].itsSizeH	= theCellSize;
			aPlacementBuffer[i].itsSizeV	= theCellSize;
		}
	}

	//	half-size cells
	//		note (h,v) indexing -- not (row,col)
	for (h = 0; h < n; h++)
	{
		for (v = 0; v < n; v++)
		{
			i = n*n + n*h + v;
			aPlacementBuffer[i].itsH		= -0.5 + h * theCellSize;
			aPlacementBuffer[i].itsV		= -0.5 + (v - APPLE_MARK_OFFSET) * theCellSize;
			aPlacementBuffer[i].itsFlip		= (md->itsGameOf.Apples2D.itsBoard[h][v] & APPLE_FLIPPED_MASK);
			aPlacementBuffer[i].itsAngle	= 0.0;
			aPlacementBuffer[i].itsSizeH	= 0.5 * theCellSize;
			aPlacementBuffer[i].itsSizeV	= 0.5 * theCellSize;
		}
	}
}
